import os
from string import Template
from TestSuite import term
from TestSuite.category import TestCategory, Category
from TestSuite.test import Test
from TestSuite.notice import *
from TestSuite.runnable import Runnable
from TestSuite.args import ArgsStore

def encode_ascii(string):
    if isinstance(string, str):
        return string.decode('utf-8')
    if isinstance(string, unicode):
        return string

class HtmlOutput:
    def __init__(self, root):
        self.root = root
        self.page = """<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">
<html xmlns="http://www.w3.org/1999/xhtml" lang="en" dir="ltr"><head><meta http-equiv="Content-Type" content="text/html; charset=UTF-8" />
<title>Tests' results</title>

<meta http-equiv="Content-Language" content="en" />
<link type="text/css" href="res/bootstrap.min.css" rel="stylesheet" />
<link type="text/css" href="res/site.min.css" rel="stylesheet" />
<link type="text/css" href="res/site.css" rel="stylesheet" />
<script type="text/javascript" src="res/jquery.min.js"></script>
<script type="text/javascript" src="res/main.js"></script>
</head>
  <body>
    <div class="container" id="main">
    <br />
    <br />
<!-- page content -->
<!-- stats & infos -->
%s
<!-- end stats & infos -->
<div class="accordion" id="acc2">
<!-- tests results -->
%s
<!-- end results -->
</div>
<!-- end page content-->
    </div>
  </body>
</html>"""

    def _percent(self, i, l):
        res = 0
        if not l:
            res = 100.0
        else:
            res = round(100.00 * (i / float(l)), 2)
        if int(res) == res:
            return int(res)
        else:
            return res

    def _panel(self, title, content, tags = 'primary'):
        template = u"""<div class="new panel panel-%s">\n  <h5 class="h panel-heading"><a class="trigger" href="#">%s</a></h5>\n  %s\n</div>\n"""
        if content:
            return template % (tags, title, '<div class="outer panel-body">\n' + content + '\n</div>')
        else:
            template = """<div class="new panel panel-%s">\n  <h5 class="h panel-heading">%s</h5>\n  %s\n</div>\n"""
            return template % (tags, title, '')

    def _badge(self, content, tags = 'primary'):
        return """<span class="badge badge-%s pull-right">%s</span>""" % (tags, content)

    def _RootCats(self, node):
        total = 1
        for n in node.subcategories:
            total += self._RootCats(n)
        return total

    def _RootTCats(self, node):
        total = len(node.testcategories)
        for n in node.subcategories:
            total += self._RootTCats(n)
        return total

    def _RootTests(self, node):
        total = 0
        for tc in node.testcategories:
            total += len(tc)
        for n in node.subcategories:
            total += self._RootTests(n)
        return total

    def _RootStats(self, node):
        # passed, failed, warn
        stats = (0, 0, 0)
        for tc in node.testcategories:
            for test in tc.tests:
                if test:
                    stats = (stats[0] + 1, stats[1], stats[2])
                else:
                    stats = (stats[0], stats[1] + 1, stats[2])
                if test.isWarning():
                    stats = (stats[0], stats[1], stats[2] + 1)
        for n in node.subcategories:
            s = self._RootStats(n)
            stats = (stats[0] + s[0],
                     stats[1] + s[1],
                     stats[2] + s[2])
        return stats

    def summary(self):
        totalpanelcontent = """<ul class="nav nav-pills nav-stacked"><li>Categories<span class="badge badge-primary pull-right">%s</span></li><li>Files<span class="badge badge-primary pull-right">%s</span></li><li>Tests<span class="badge badge-primary pull-right">%s</span></li></ul>""" % \
                (self._RootCats(self.root) - 1, self._RootTCats(self.root), self._RootTests(self.root))
        cpcc = ''
        for cat in self.root.subcategories:
            cpcc += """<li>%s<span class="badge badge-%s pull-right">%s%%</span></li>""" % (cat.name, 'success' if cat else 'danger', self._percent(cat.success(), len(cat)))
        for cat in self.root.testcategories:
            cpcc += """<li>%s<span class="badge badge-%s pull-right">%s%%</span></li>""" % (cat.name, 'success' if cat else 'danger', self._percent(cat.success(), len(cat)))
        categoriespanelcontet = """<ul class="nav nav-pills nav-stacked">%s</ul>""" % cpcc
        testspanelcontent = """<ul class="nav nav-pills nav-stacked"><li>Passed<span class="badge badge-success pull-right">%d</span></li><li>Failed<span class="badge badge-danger pull-right">%d</span></li><li>Warnings<span class="badge badge-warning pull-right">%d</span></li></ul>""" % self._RootStats(self.root)

        return """<div class="row"><div class="col-md-4">%s</div><div class="col-md-4">%s</div><div class="col-md-4">%s</div></div>""" % \
                (self._panel('Total', totalpanelcontent, 'info'), self._panel('Categories', categoriespanelcontet, 'info'), self._panel('Tests', testspanelcontent, 'info'))
        

    def _writePrePost(self, runn):
        content_wrapper = """<div class="inner content-area">%s</div>"""
        init_out = ''
        init_out += '<pre>$ %s\n...\n$ echo $?\n%s</pre>\n' % (encode_ascii(runn.cmd), runn.pid.returncode)
        if runn.stdout:
            init_out += self._panel('stdout', content_wrapper % encode_ascii(runn.stdout), 'info')
        if runn.stderr:
            init_out += self._panel('stderr', content_wrapper % encode_ascii(runn.stderr), 'danger')

        return encode_ascii(init_out)

    def _writeTestSubpanel(self, test, title, key, legend = True):
        content_wrapper = u"""<div class="inner content-area">%s</div>"""
        if legend:
            content_wrapper += '<br />' + self._panel('Legend', content_wrapper % u"""<span class="diff-delete">expected</span> <span class="diff-insert">having</span> <span class="diff-equal">equal</span>""", 'info panel-xs')
        diff = test.diff.getDiffHtml(key)
        if diff:
            return self._panel(title, content_wrapper % encode_ascii(test.diff.getDiffHtml(key)), 'success' if test.diff.isValid(key) else 'danger')
        else:
            return self._panel(title, '', 'default')


    def _getWarnings(self, test):
        output = ''
        warns = test.getWarnings()
        template = u"""<div class="alert alert-warning"><strong>Warning: </strong>%s %s</div>\n"""
        for w in warns.keys():
            if isinstance(warns[w], list):
                for i in warns[w]:
                    output += template % (encode_ascii(i[1]), '')
            elif isinstance(warns[w], Runnable):
                if warns[w].pid and warns[w].pid.returncode:
                    output += template % (encode_ascii(w), u'exited with code ' + str(warns[w].pid.returncode))
                if warns[w].stderr:
                    output += template % (encode_ascii(w), u'printed text on stderr')
        return output

    def _writeTest(self, test):
        output = ''
        tag = u'success' if test else u'danger'
        warn = False
        if test.isWarning():
            warn = True
            tag += '-warning'
            output += self._getWarnings(test)

        # write diffs & details
        if test.diff.isValid('return'):
            output += '<pre>$ %s\n...\n$ echo $?\n%s</pre>\n' % (test.expected.cmd.decode('utf-8'), test.expected.val)
        else:
            output += '<pre>$ %s</pre>\n' % test.expected.cmd.decode('utf-8')
        title = encode_ascii(test.name)
        if test.killed:
            title += self._badge('KILLED', 'danger')
        if (test.isWarning()):
            title += self._badge('WARNING', 'warning')

        if test.pretest:
            output += self._panel(u'pre-test script', self._writePrePost(test.pretest), u'warning' if test.pretest.has_warning() else u'info')
        if test.posttest:
            output += self._panel(u'post-test script', self._writePrePost(test.posttest), u'warning' if test.posttest.has_warning() else u'info')
        if test.killed:
            output += self._panel(u'time', u'Test last more than %s seconds' % test.expected.time, u'danger')
        if test.expected.stdin:
            output += self._panel(u'input', u"""<pre><div class="inner content-area">%s</div></pre>""" % encode_ascii(test.expected.stdin), 'info')

        if not test.diff.isValid('return'):
            output += self._writeTestSubpanel(test, 'return code', 'return', legend=False)
        output += self._writeTestSubpanel(test, 'stdout', 'stdout')
        output += self._writeTestSubpanel(test, 'stderr', 'stderr')
        return self._panel(title, encode_ascii(output), tag)

    def explore(self, node):
        content = ''
        if isinstance(node, Category):
            if len(node) == 0:
                return ''
            for tcat in node.testcategories:
                success = self._percent(tcat.success(), len(tcat))
                badge = """<span class="badge badge-%s pull-right">%s%%</span>""" % ('danger' if success < 100 else 'success', success)
                content += self._panel(tcat.name.decode('utf-8') + badge, self.explore(tcat))
            for cat in node.subcategories:
                success = self._percent(cat.success(), len(cat))
                badge = """<span class="badge badge-%s pull-right">%s%%</span>""" % ('danger' if success < 100 else 'success', success)
                content += self._panel(cat.name.decode('utf-8') + badge, self.explore(cat))
        elif isinstance(node, TestCategory):
            if not len(node.tests):
                return ''
            for test in node.tests:
                content += self._writeTest(test)

        return content

    def render(self):
        return self.page % (self.summary() + '\n<br />\n', self.explore(self.root))

    def write(self, path):
        if ArgsStore.output:
            path = os.path.join(ArgsStore.output, path)
        f = None
        try:
            f = file(path, 'w')
            f.write(self.render().encode('utf-8'))
            return True
        except IOError:
            print term.format('Error', term.red) + ' while writing to ' + term.format(path, term.bold)
            return False
