import subprocess
from string import Template
from TestSuite import ConfigReader
from TestSuite.args import *
import TestSuite.google_diffutils as diffutils
from TestSuite.runnable import *
from TestSuite import notice
from TestSuite import utils

dmp = diffutils.diff_match_patch()
dmp.Diff_Timeout = 0

class ExpectValue:
    def __init__(self):
        self.details = {}

    # expected : Expect, having:Runnable
    def matchAll(self, expected, having):
        if expected.stdout != '*':
            self.details['stdout'] = dmp.diff_main(expected.stdout if expected.stdout else '', having.stdout)
        else:
            self.details['stdout'] = dmp.diff_main(having.stdout, having.stdout)
        if expected.stderr != '*':
            self.details['stderr'] = dmp.diff_main(expected.stderr if expected.stderr else '', having.stderr)
        else:
            self.details['stderr'] = dmp.diff_main(having.stderr, having.stderr)

        returncode = str(having.pid.returncode) if having.pid else '127'
        if expected.val >= 0:
            self.details['return'] = dmp.diff_main(str(expected.val), returncode)
        else:
            self.details['return'] = dmp.diff_main(returncode, returncode)
    
        self.cleanSemantics()
        return self.isValid('stdout') and self.isValid('stderr') and self.isValid('return')

    def cleanSemantics(self):
        for k in self.keys():
            dmp.diff_cleanupSemantic(self[k])

    def getDiffHtml(self, k):
        return dmp.diff_prettyHtml(self[k])

    def isValid(self, key):
        return not(len(self.details[key])) or (len(self.details[key]) == 1 and self.details[key][0][0] == 0)
    # special methods to use ExpectValue as container
    def __setitem__(self, key, value):
        self.details[key] = value
    def __getitem__(self, key):
        return self.details[key]
    def __iter__(self):
        return self.details.__iter__()
    def iterkeys(self):
        return self.details.iterkeys()
    def iteritems(self):
        return self.details.iteritems()
    # overload dict methods
    def keys(self):
        return self.details.keys()
    def values(self):
        return self.details.values()

class Expect:
    def __init__(self, config, path, parent):
        self.config = config
        self.path = path
        self.parent = parent

        name = self.parent.name

        self.before = ConfigReader.getKeyIfExists(config, name, 'before')
        self.beforeout = ConfigReader.getKeyIfExists(config, name, 'beforeout')
        if self.beforeout:
            self.beforeout = Template(self.beforeout).safe_substitute({'cwd':self.path})
        self.after = ConfigReader.getKeyIfExists(config, name, 'after')

        self.val = ConfigReader.getKeyIfExists(config, name, 'return')
        self.stdout = utils.strip_lines(ConfigReader.getKeyIfExists(config, name, 'stdout'))
        self.stderr = utils.strip_lines(ConfigReader.getKeyIfExists(config, name, 'stderr'))
        self.stdin  = ConfigReader.getKeyIfExists(config, name, 'stdin')
        self.fstdout = ConfigReader.getKeyIfExists(config, name, 'fstdout') # stdout from file
        self.fstderr = ConfigReader.getKeyIfExists(config, name, 'fstderr') # stderr from file
        self.cmd = Template(config.get(name, 'cmd')).safe_substitute({'cwd':self.path, 'bout':self.beforeout if self.beforeout else ''})

        self.time = ArgsStore.time if ArgsStore.time else ConfigReader.getKeyIfExists(config, name, 'time')
        if self.time:
            self.time = float(self.time)

        if self.val is None:
            self.val = 0

        if self.before:
            self.before = Template(self.before).safe_substitute({'cwd':self.path})
        if self.after:
            self.after = Template(self.after).safe_substitute({'cwd':self.path})

    def openTestFiles(self):
        if self.fstdout:
            data = self._from_file(self.fstdout)
            self.stdout = data if data else self.stdout
        if self.fstderr:
            data = self._from_file(self.fstderr)
            self.stderr = data if data else self.stderr

    def _from_file(self, path):
        path = Template(path).safe_substitute({'cwd':self.path})
        f = None
        try:
            f = open(path, 'r')
            return utils.strip_lines(''.join(f.readlines()))
        except:
            notice.log(notice.warn, self.parent, 'Unable to open file: %s' % path)
            return None

class Test:
    """
    A single Test
    """
    def __init__(self, name, config, category):
        self.name = name
        self.category = category
        self.expected = Expect(config, category.path, self)
        self.diff = ExpectValue()

        self.passed = False
        self.killed = False

        self.pretest = None
        self.posttest = None

    def __nonzero__(self):
        return self.passed and not self.killed
    def __str__(self):
        return self.name

    def _pretest(self):
        if self.expected.before:
            runn = Runnable(self.expected.before, file_out = self.expected.beforeout)
            runn.run()
            return runn
        else:
            return None

    def _posttest(self):
        if self.expected.after:
            runn = Runnable(self.expected.after)
            runn.run()
        else:
            return None
        return runn

    def run(self):
        # run before and check stderr (treat as warning)
        self.pretest = self._pretest()
        # eventually update stdout and stderr with files
        # updating only now so that we can use files generated by 'pretest' script
        self.expected.openTestFiles()
        # match values with self.diff
        test = Runnable(self.expected.cmd, self.expected.stdin, self.expected.time)
        self.killed = test.run()
        if self.diff.matchAll(self.expected, test):
            self.passed = True
        # run after and check stderr (treat as warning)
        self.posttest = self._posttest()

    def getWarnings(self):
        rv = {}
        if self.pretest and (self.pretest.stderr or self.pretest.pid.returncode):
            rv['pre-test'] = self.pretest
        if self.posttest and (self.posttest.stderr or self.posttest.pid.returncode):
            rv['post-test'] = self.posttest
        if notice.Notices.notices.has_key(self):
            rv['notices'] = notice.Notices.notices[self]
        return rv

    def isWarning(self):
        if self.pretest:
            if self.pretest.stderr or self.pretest.pid.returncode:
                return True
        if self.posttest:
            if self.posttest.stderr or self.posttest.pid.returncode:
                return True
        if notice.Notices.notices.has_key(self):
            return True
        return False
